<?php
// Add this at the very top to ensure errors are visible
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session (with check to avoid duplicate session starts)
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verify config.php is included properly
if (!file_exists('config.php')) {
    die("Configuration file missing");
}
include 'config.php';

// Enable mysqli error reporting
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

try {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $errors = [];

        // Verify all required fields exist in POST
        $required = ['fullName', 'nationalID', 'governorate', 'directorate'];
        foreach ($required as $field) {
            if (!isset($_POST[$field])) {
                $errors[] = "حقل مطلوب مفقود: $field";
            }
        }

        if (!empty($errors)) {
            throw new Exception(implode("\n", $errors));
        }

        // Input sanitization
        $fullName = trim($_POST['fullName']);
        $nationalID = trim($_POST['nationalID']);
        $governorate = (int)$_POST['governorate'];
        $directorate = (int)$_POST['directorate'];
        $email = isset($_POST['email']) ? trim($_POST['email']) : null;
        $phone = isset($_POST['phone']) ? trim($_POST['phone']) : null;

        // Validation rules
        if (empty($fullName)) $errors[] = "الاسم الكامل مطلوب.";
        
        if (empty($nationalID)) {
            $errors[] = "الرقم القومي مطلوب.";
        } elseif (strlen($nationalID) !== 14 || !ctype_digit($nationalID)) {
            $errors[] = "الرقم القومي يجب أن يتكون من 14 رقمًا فقط.";
        }

        if ($governorate <= 0) $errors[] = "المحافظة مطلوبة بشكل صحيح.";
        if ($directorate <= 0) $errors[] = "المديرية مطلوبة بشكل صحيح.";
        
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "صيغة البريد الإلكتروني غير صحيحة.";
        }
        
        if (!empty($phone) && !preg_match('/^\d+$/', $phone)) {
            $errors[] = "رقم الهاتف يجب أن يحتوي على أرقام فقط.";
        }

        if (!empty($errors)) {
            throw new Exception(implode("\n", $errors));
        }

        // Verify database connection
        if (!$conn) {
            throw new Exception("لا يمكن الاتصال بقاعدة البيانات");
        }

        // Check if governorate and directorate exist
        $checkGov = $conn->prepare("SELECT 1 FROM governorates WHERE GovernorateID = ?");
        $checkGov->bind_param("i", $governorate);
        $checkGov->execute();
        if (!$checkGov->get_result()->fetch_assoc()) {
            throw new Exception("المحافظة المحددة غير موجودة");
        }

        $checkDir = $conn->prepare("SELECT 1 FROM directorates WHERE DirectorateID = ? AND GovernorateID = ?");
        $checkDir->bind_param("ii", $directorate, $governorate);
        $checkDir->execute();
        if (!$checkDir->get_result()->fetch_assoc()) {
            throw new Exception("المديرية المحددة غير موجودة في هذه المحافظة");
        }

        // Database operations
        $sql = "INSERT INTO users (NationalID, FullName, GovernorateID, DirectorateID, Email, Phone) 
                VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception("تحضير الاستعلام فشل: " . $conn->error);
        }
        
        $stmt->bind_param("ssiiss", $nationalID, $fullName, $governorate, $directorate, $email, $phone);
        
        if ($stmt->execute()) {
            $_SESSION['success'] = true;
            header("Location: login.php?success=1");
            exit();
        } else {
            throw new Exception("فشل تنفيذ الاستعلام");
        }
    }
} catch (mysqli_sql_exception $e) {
    $errorMsg = "حدث خطأ في قاعدة البيانات. الرجاء المحاولة لاحقاً.";
    if ($e->getCode() == 1062) {
        $errorMsg = "الرقم القومي مسجل مسبقاً.";
    }
    error_log("Database error: " . $e->getMessage());
    $_SESSION['errors'] = [$errorMsg];
    $_SESSION['old'] = $_POST;
    header("Location: register.php");
    exit();
} catch (Exception $e) {
    error_log("General error: " . $e->getMessage());
    $_SESSION['errors'] = explode("\n", $e->getMessage());
    $_SESSION['old'] = $_POST;
    header("Location: register.php");
    exit();
}